//*************************************************************************************************
//
//	Description:
//		garage_lensflare.fx - A lensflare on a stick for the garage scene.
//
//	<P> Copyright (c) 2009 Slightly Mad Studios Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     12/06/2009  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// A static multiplier applied to the alpha of each pixel (with platform-specific values)
#if defined( _PS3_ )

#define OVERALL_FADE_MULTIPLIER	0.5f

#elif defined( _XBOX_ )

#define OVERALL_FADE_MULTIPLIER	0.5f

#else

#define OVERALL_FADE_MULTIPLIER	0.5f

#endif

// A multiplier applied to the alpha of each pixel based on the formula: alpha = alpha x ( 1 - (cam_height x HEIGHT_FADE_MULTIPLIER) )
#define HEIGHT_FADE_MULTIPLIER	0.62f

// The lowest the height fade multiplier is allowed to go, to avoid high cams making the object fade too much
#define HEIGHT_MIN_FADE					0.03f

// Size of the quad in world units
#define QUAD_SIZE										7.0f

// The car is centred on (0,0,0), so the object is placed opposite the camera at (camera_distance_from_origin x DISTANCE_MULTIPLIER) away from the origin
#define DISTANCE_MULTIPLIER		2.0f

// Once placed opposite the camera, the object is moved vertically by this amount
#define VERTICAL_OFFSET ( QUAD_SIZE * 0.55f )



//-----------------------------------------------------------------------
//
// Input parameters
//

#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif

//
// Transforms
//

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

SHARE_PARAM float4x4 view : View
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

SHARE_PARAM float4x4 projMatrix : Projection
<
	bool appEdit = false;
	bool export = false;
>;


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Second UV holds the billboarding pivot offsets
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
	bool use3dMapping = true;
> = 0;
#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
	bool export = true;
>;



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;								// Object space position
	float2 texCoord : TEXCOORD0;							// UV channel 1 texture coord
	float3 billboardPivot: TEXCOORD1;
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;						// View-coords position
	float2 texCoord	: TEXCOORD0;						// UV coords for texture channel 0
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//


VSOUTPUT GarageLensFlareVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

	// Calculate the world coords pivot position from the camera position (it's directly opposite the camera, a set number of units from the origin)
#ifdef _3DSMAX_
	float4 cameraPos = viewI[ 3 ];
#else
	float4 cameraPos = float4( worldCameraPos.xyz, 1.0f );
#endif
	float4 worldPivotPos = float4( -cameraPos.xyz * DISTANCE_MULTIPLIER, 1.0f );
	float2 quadSize = float2( QUAD_SIZE, QUAD_SIZE );
	float4 pivotInView = mul( worldPivotPos, view );
	pivotInView.y += VERTICAL_OFFSET;

	// Calculate the vertex position in view coords
	float4 cornerOffset = float4( ( _input.texCoord.x - 0.5f ) * quadSize.x, ( 0.5f - _input.texCoord.y ) * quadSize.y, 0.0f, 0.0f );
	float4 viewPos = pivotInView + cornerOffset;

	// Calculate clip-space position of the vertex
	_output.position = mul( viewPos, projMatrix );

	_output.texCoord = _input.texCoord;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

// Input structure
struct PSINPUT
{
	float2 texCoord	: TEXCOORD0;		// UV coords for texture channel 0
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT GarageLensFlareFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read texture
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour;

	// Calculate camera height (assume floor is at 0.0)
#ifdef _3DSMAX_
	float camYPos = 0.0f;
#else
	float camYPos = worldCameraPos.y;
#endif

	float fade = OVERALL_FADE_MULTIPLIER * max( 1.0f - saturate( camYPos * HEIGHT_FADE_MULTIPLIER ), HEIGHT_MIN_FADE );

	accumulatedColour.a *= fade;

	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique GarageLensFlare
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GarageLensFlare";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx GarageLensFlareVertexShader();
		PixelShader = compile sce_fp_rsx GarageLensFlareFragmentShader();
#else
		VertexShader = compile vs_3_0 GarageLensFlareVertexShader();
		PixelShader = compile ps_3_0 GarageLensFlareFragmentShader();
#endif
	}
}
